<?php
/* --------------------------------------------------------------
   TrackingCodeSorting.php 2020-04-16
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2019 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

declare(strict_types=1);

namespace Gambio\Admin\TrackingCode\SqlCriteria;

use Gambio\Core\Criteria\SqlSorting;
use Webmozart\Assert\Assert;

/**
 * Class TrackingCodeSorting
 *
 * @package Gambio\Admin\TrackingCode\SqlCriteria
 */
class TrackingCodeSorting implements SqlSorting
{
    /**
     * @var string
     */
    private $attribute;
    
    /**
     * @var string
     */
    private $order;
    
    /**
     * @var string[]
     */
    private static $attributeColumnMapping = [
        'id'                    => 'orders_parcel_tracking_code_id',
        'orderId'               => 'order_id',
        'code'                  => 'tracking_code',
        'languageId'            => 'language_id',
        'parcelService.id'      => 'parcel_service_id',
        'parcelService.name'    => 'parcel_service_name',
        'parcelService.url'     => 'url',
        'parcelService.comment' => 'comment',
        'createdOn'             => 'creation_date',
    ];
    
    
    /**
     * TrackingCodeSorting constructor.
     *
     * @param string $attribute
     * @param string $order
     */
    private function __construct(string $attribute, string $order)
    {
        
        $this->attribute = $attribute;
        $this->order     = strtolower($order);
    }
    
    
    /**
     * @param string $attribute
     * @param string $order
     *
     * @return TrackingCodeSorting
     */
    public static function create(string $attribute, string $order = 'ASC'): TrackingCodeSorting
    {
        $attributes = array_keys(self::$attributeColumnMapping);
        Assert::oneOf($attribute,
                      $attributes,
                      'Provided attribute needs to be one of: ' . implode(', ', $attributes) . '; Got: ' . $attribute);
        Assert::oneOf(strtolower($order),
                      ['asc', 'desc'],
                      'Provided order needs to be one of: asc, desc; Got: ' . $order);
        
        return new self($attribute, $order);
    }
    
    
    /**
     * @return string
     */
    public function attribute(): string
    {
        return $this->attribute;
    }
    
    
    /**
     * @return string
     */
    public function order(): string
    {
        return $this->order;
    }
    
    
    /**
     * @return string
     */
    public function column(): string
    {
        return self::$attributeColumnMapping[$this->attribute];
    }
}